//
// TypeHandler.h
//
// $Id: //poco/1.4/Data/include/Poco/Data/TypeHandler.h#1 $
//
// Library: Data
// Package: DataCore
// Module:  TypeHandler
//
// Definition of the TypeHandler class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_TypeHandler_INCLUDED
#define Data_TypeHandler_INCLUDED


#include "Poco/Data/Data.h"
#include "Poco/Data/AbstractBinder.h"
#include "Poco/Data/AbstractExtractor.h"
#include "Poco/Data/AbstractPreparation.h"
#include "Poco/Tuple.h"
#include <cstddef>


namespace Poco {
namespace Data {


template <class T>
class TypeHandler
	/// Converts Rows to a Type and the other way around. Provide template specializations to support your own complex types.
	///
	/// Take as example the following (simplified) class:
	///    class Person
	///    {
	///    private:
	///        std::string _lastName;
	///        std::string _firstName;
	///        int         _age;
	///        [....] // public set/get methods, a default constructor, optional < operator (for set, multiset) or function operator (for map, multimap)
	///    };
	///
	/// The TypeHandler must provide a costum bind, size, prepare and extract method:
	///    
	///    template <>
	///    class TypeHandler<struct Person>
	///    {
	///    public:
	///        static std::size_t size()
	///        {
	///            return 3; // lastName + firstname + age occupy three columns
	///        }
	///    
	///        static void bind(std::size_t pos, const Person& obj, AbstractBinder* pBinder)
	///        {
	///            // the table is defined as Person (LastName VARCHAR(30), FirstName VARCHAR, Age INTEGER(3))
	///            // Note that we advance pos by the number of columns the datatype uses! For string/int this is one.
	///            poco_assert_dbg (pBinder != 0);
	///            TypeHandler<std::string>::bind(pos++, obj.getLastName(), pBinder);
	///            TypeHandler<std::string>::bind(pos++, obj.getFirstName(), pBinder);
	///            TypeHandler<int>::bind(pos++, obj.getAge(), pBinder);
	///        }
	///    
	///        static void prepare(std::size_t pos, const Person& obj, AbstractPreparation* pPrepare)
	///        {
	///            // the table is defined as Person (LastName VARCHAR(30), FirstName VARCHAR, Age INTEGER(3))
	///            poco_assert_dbg (pPrepare != 0);
	///            TypeHandler<std::string>::prepare(pos++, obj.getLastName(), pPrepare);
	///            TypeHandler<std::string>::prepare(pos++, obj.getFirstName(), pPrepare);
	///            TypeHandler<int>::prepare(pos++, obj.getAge(), pPrepare);
	///        }
	///    
	///        static void extract(std::size_t pos, Person& obj, const Person& defVal, AbstractExtractor* pExt)
	///        {
	///            // defVal is the default person we should use if we encunter NULL entries, so we take the individual fields
	///            // as defaults. You can do more complex checking, ie return defVal if only one single entry of the fields is null etc...
	///            poco_assert_dbg (pExt != 0);
	///            std::string lastName;
	///            std::string firstName;
	///            int age = 0;
	///            // the table is defined as Person (LastName VARCHAR(30), FirstName VARCHAR, Age INTEGER(3))
	///            TypeHandler<std::string>::extract(pos++, lastName, defVal.getLastName(), pExt);
	///            TypeHandler<std::string>::extract(pos++, firstName, defVal.getFirstName(), pExt);
	///            TypeHandler<int>::extract(pos++, age, defVal.getAge(), pExt);
	///            obj.setLastName(lastName);
	///            obj.setFirstName(firstName);
	///            obj.setAge(age);
	///        }
	///    };
	///
	/// Note that the TypeHandler template specialization must always be declared in the namespace Poco::Data.
	/// Apart from that no further work is needed. One can now use Person with into and use clauses.
{
public:
	static void bind(std::size_t pos, const T& obj, AbstractBinder* pBinder)
	{
		poco_assert_dbg (pBinder != 0);
		pBinder->bind(pos, obj);
	}

	static std::size_t size()
	{
		return 1;
	}

	static void extract(std::size_t pos, T& obj, const T& defVal, AbstractExtractor* pExt)
	{
		poco_assert_dbg (pExt != 0);
		if (!pExt->extract(pos, obj))
			obj = defVal;
	}

	static void prepare(std::size_t pos, const T& obj, AbstractPreparation* pPrepare)
	{
		poco_assert_dbg (pPrepare != 0);
		pPrepare->prepare(pos, obj);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator = (const TypeHandler&);
};


/// Poco::Tuple TypeHandler specializations

// define this macro to nothing for smaller code size
#define POCO_TUPLE_TYPE_HANDLER_INLINE inline

template <typename TupleType, typename Type, int N>
POCO_TUPLE_TYPE_HANDLER_INLINE
void tupleBind(std::size_t& pos, TupleType tuple, AbstractBinder* pBinder)
{
	TypeHandler<Type>::bind(pos, tuple.template get<N>(), pBinder);
	pos += TypeHandler<Type>::size();
}


template <typename TupleType, typename Type, int N>
POCO_TUPLE_TYPE_HANDLER_INLINE
void tuplePrepare(std::size_t& pos, TupleType tuple, AbstractPreparation* pPrepare)
{
	TypeHandler<Type>::prepare(pos, tuple.template get<N>(), pPrepare);
	pos += TypeHandler<Type>::size();
}


template <typename TupleType, typename DefValType, typename Type, int N>
POCO_TUPLE_TYPE_HANDLER_INLINE
void tupleExtract(std::size_t& pos, TupleType tuple, DefValType defVal, AbstractExtractor* pExt)
{
	TypeHandler<Type>::extract(pos, tuple.template get<N>(), defVal.template get<N>(), pExt);
	pos += TypeHandler<Type>::size();
}



template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16, 
	class T17, 
	class T18, 
	class T19>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T14, 14>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T15, 15>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T16, 16>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T17, 17>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T18, 18>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T19, 19>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T14, 14>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T15, 15>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T16, 16>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T17, 17>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T18, 18>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T19, 19>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size() +
			TypeHandler<T14>::size() +
			TypeHandler<T15>::size() +
			TypeHandler<T16>::size() +
			TypeHandler<T17>::size() +
			TypeHandler<T18>::size() +
			TypeHandler<T19>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T14, 14>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T15, 15>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T16, 16>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T17, 17>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T18, 18>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T19, 19>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16, 
	class T17, 
	class T18>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T14, 14>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T15, 15>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T16, 16>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T17, 17>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T18, 18>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T14, 14>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T15, 15>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T16, 16>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T17, 17>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T18, 18>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size() +
			TypeHandler<T14>::size() +
			TypeHandler<T15>::size() +
			TypeHandler<T16>::size() +
			TypeHandler<T17>::size() +
			TypeHandler<T18>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T14, 14>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T15, 15>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T16, 16>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T17, 17>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T18, 18>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16, 
	class T17>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T14, 14>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T15, 15>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T16, 16>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T17, 17>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T14, 14>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T15, 15>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T16, 16>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T17, 17>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size() +
			TypeHandler<T14>::size() +
			TypeHandler<T15>::size() +
			TypeHandler<T16>::size() +
			TypeHandler<T17>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T14, 14>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T15, 15>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T16, 16>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T17, 17>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T14, 14>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T15, 15>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T16, 16>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T14, 14>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T15, 15>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T16, 16>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size() +
			TypeHandler<T14>::size() +
			TypeHandler<T15>::size() +
			TypeHandler<T16>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T14, 14>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T15, 15>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T16, 16>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T14, 14>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T15, 15>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T14, 14>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T15, 15>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size() +
			TypeHandler<T14>::size() +
			TypeHandler<T15>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T14, 14>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T15, 15>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T14, 14>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T14, 14>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size() +
			TypeHandler<T14>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T14, 14>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T13, 13>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T13, 13>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size() +
			TypeHandler<T13>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T13, 13>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T12, 12>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T12, 12>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size() +
			TypeHandler<T12>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T12, 12>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T11, 11>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T11, 11>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size() +
			TypeHandler<T11>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T11, 11>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T10, 10>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T10, 10>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size() +
			TypeHandler<T10>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T10, 10>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8, class T9>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T9, 9>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T9, 9>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size() +
			TypeHandler<T9>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T9, 9>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T8, 8>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T8, 8>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size() +
			TypeHandler<T8>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T8, 8>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6, class T7>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T7, 7>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T7, 7>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size() +
			TypeHandler<T7>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T7, 7>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T6, 6>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T6, 6>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size() +
			TypeHandler<T6>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T6, 6>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T5, 5>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T5, 5>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size() +
			TypeHandler<T5>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T5, 5>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T4, 4>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T4, 4>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size() +
			TypeHandler<T4>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T4, 4>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T3, 3>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T3, 3>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size() +
			TypeHandler<T3>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T3, 3>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2>
class TypeHandler<Poco::Tuple<T0, T1, T2, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T2, 2>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T2, 2>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size() +
			TypeHandler<T2>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T2, 2>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1>
class TypeHandler<Poco::Tuple<T0, T1, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
		tupleBind<TupleConstRef, T1, 1>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
		tuplePrepare<TupleConstRef, T1, 1>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size() +
			TypeHandler<T1>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
		tupleExtract<TupleRef, TupleConstRef, T1, 1>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0>
class TypeHandler<Poco::Tuple<T0, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		tupleBind<TupleConstRef, T0, 0>(pos, tuple, pBinder);
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		tuplePrepare<TupleConstRef, T0, 0>(pos, tuple, pPrepare);
	}

	static std::size_t size()
	{
		return TypeHandler<T0>::size();
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, 
		AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		tupleExtract<TupleRef, TupleConstRef, T0, 0>(pos, tuple, defVal, pExt);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


} } // namespace Poco::Data


#endif // Data_TypeHandler_INCLUDED
